#!/usr/local/bin/R

#*******************************************************************************
#***************                                                    ************
#***************   Yipeng WANG's Microarray Data Analysis Script    ************
#***************                                                    ************
#*******************************************************************************

#******************** Function of this R script  *******************************
#** This script is used for microarray data analysis using limma package.
#** FDR is estimated.
#** CGH analysis.
#** PCA normalization.
#*******************************************************************************

#####################################################
############# experiment parameters  ################
#####################################################

## Xiaoqin: these are the parameters USERS need to input

as.CGI = TRUE #FALSE

	#New Parameters.
	#Radio Group(Input microarray file format): "agilent", "arrayvision", "genepix", "imagene", "quantarray"(default), "smd", "spot" or "spot.close.open". 
	#Radio Group(gene list file): "GAL (default)", "User Defined", "none";
	#Radio Group(Background Subtraction): "none", "subtract(dedault)", "half", "minimum", "movingmin", "edwards" or "normexp"
	#Radio Group(Spot Quality Weight): NULL(default)", "wtarea(SPOT output)", "wtflags(GenePix output)", "wtIgnore.Filter(QuantArray output)"
	#Radio Group(WithinArray Normalization): "none", "median", "loess", "printtiploess"(default), "composite" and "robustspline".
	# A "broswer" edit box for composite normalization.
	#Radio Group(BetweenArray Normalization): "none", "scale"(default), "quantile", "Aquantile", "Gquantile", "Rquantile", "Tquantile", "vsn"
	#Radio Group(Output File Format): Rank by SpotID (default)/Rank by p value
	#Radio Group(Final results, include:M-A plot, M-B plot, p value distribution): Yes(default)/No
	#Radio Group(Array Image Plot): Yes(default)/No
	#Radio Group(M-A plot, All slides, before and after withinArray Normalization): Yes(default)/No
	#Radio Group(M Scale plot, All slides, before and after withinArray Normalization): Yes(default)/No
	#Radio Group(Box plot, All slides, before and after betweenArray Normalization): Yes(default)/No
	
default.params <- list(
	#PrintTip.Chart.Title = "PrintTip_Plot_PrintTip_Norm.png",
	Final.Chart.Title = "Final_Results_Plot",
	Scale.Chart.Title = "Box_Plot_Scale_Norm.png",
	Array.Chart.Title = "imageplot.slide.",
	Printtip.Chart.Title = 'printtiploessplot.slides.',
	Density.Chart.Title = 'densityplot.png',
	MA.Chart.Title = 'plotMA.slides.',
	Location.Chart.Title = 'plot.CGH.two.samples',

	result.norm.data = "output.normalized.data.txt",
	result.analyzed.data = "output.analyzed.data",
	
	source.dir = "sources",
	
	############### data might come from web users ################
	
	# directories and files
	graphDir = paste(getwd(),"/ResultsData_Output/Results_Charts/", sep=''), #note the tailing "/" !
	raw.data.dir = "RawData_Input",
	
	result.data.dir = "ResultsData_Output",
	chart.dir = "ResultsData_Output/Results_Charts",
	
	design.file = "RawData_Input/design.txt",
	gal.file.name = "RawData_Input/11th.gal",
	location.file = "RawData_Input/genes_chr_hpaii.txt",
	targets.file = "Targets.txt",
	spot.types.files = "RawData_Input/SpotTypes.txt",
	controlspots = "",
	sample.info.file = "RawData_Input/Samples_Info.txt", # didn't used
	
	Num.of.Dup = 1,
	spacing=1,
	gal = "GAL",
	design.ch1 = c(), #c("ctrl","ctrl","ctrl","ctrl"),
	design.ch2 = c(), #c("exp","exp","exp","exp"),
	design.ref = FALSE,
	simple.design = FALSE, #TRUE, # should be calculated with web data
	contrast <- c("exp1-ctrl", "ctrl-exp1"),
	contrast.design <- list(c("exp1","ctrl"),c("ctrl","exp1")),
	
	work.dir = "",  #changed by ywang work.dir = ".",
	script.dir = ".",
	path.sep = '/',
	
	use.targets.file = TRUE,
	slides = list(), # list of intensity files
	
	MA.File.Format = c("quantarray"),
	WithinArray.Nor = c("printtiploess"), # should be 'print-tip lowess'?
	BetweenArra.Nor = c('quantile'), #c("scale"),
	
	BG.Sub = "subtract",
	wt.fun = NULL,
	wt.para = 0, #100,
	spot.types = FALSE, # can be calculated from its file name, CANNOT be TRUE !!!!!!!!
	analysis = 'linear model analysis',
	Output.Format = c("SpotID"),
	
	plot.analyzed.results = TRUE,
	plot.arrayimage = TRUE,
	plot.printtiploess = TRUE,
	plot.box = TRUE,
	plot.density = TRUE,
	plot.MA = TRUE,
	plot.cgh = TRUE,
	
	genome.sorting = TRUE, # CGH analysis
	genomes = "human",
	genome.no = 10,
	span = 0.8, 
	chr.thre.1 = 1.8, 
	chr.thre.2 = 0.001,
	chr.lim.1 = 1.5, 
	chr.lim.2 = 3,

	req_name = ''
)

wd <- getwd() #added by xxq

source('InitParams.R')

setwd(script.dir) #added by xxq

addNamePrefix <- function() {
	if (req_name != '') {
		Final.Chart.Title <<- paste(req_name, "_", Final.Chart.Title, sep='')
		Scale.Chart.Title <<- paste(req_name, "_", Scale.Chart.Title, sep='')
		Array.Chart.Title <<- paste(req_name, "_", Array.Chart.Title, sep='')
		Printtip.Chart.Title <<- paste(req_name, "_", Printtip.Chart.Title, sep='')
		Density.Chart.Title <<- paste(req_name, "_", Density.Chart.Title, sep='')
		MA.Chart.Title <<- paste(req_name, "_", MA.Chart.Title, sep='')
		Location.Chart.Title <<- paste(req_name, "_", Location.Chart.Title, sep='')
		result.norm.data <<- paste(req_name, "_", result.norm.data, sep='')
		result.analyzed.data <<- paste(req_name, "_", result.analyzed.data, sep='')
		}
	}

mypaste <- function(src, i, max_num, sep='') {
	if (i>0 && max_num>0 && i<max_num){
		wid <- as.integer(log10(max_num))+1
		iwid <- as.integer(log10(length(i)))+1
		owid <- wid - iwid
		zs <- ''
		if (owid>0){
			for (k in 1:owid) zs <- paste(zs, '0', sep='')
			}
		zs <- paste(src, zs, i, sep=sep)
	}else zs <- paste(src, i , sep=sep)
	zs
	}

mycontrastname <- function(contra, format){
	contrast.name <- c()
    if(identical(format, "list")){
		for(i in 1:length(contra)){
			contrast.name <- c(contrast.name, paste(contra[[i]][1],".", contra[[i]][2],sep=""))
		}
	}else{
		contrast.name <- contra[2: length(contra)]
	}
	return(contrast.name)
}


#####################################################
########### load library and source code  ###########
#####################################################

library(limma);
library(sma);
library(vsn);

source('normPCA.R');
setwd(source.dir);
source("splosh-code3_000.R");
source("CGH_v01.R");
source("myMakeContrast.R");
source("mkDesign.R");
source("plot-results.R");
source("read_imagefile.R");
setwd(wd);

#####################################################
########### read exported microarray data  ##########
#####################################################

readIntensity <- function() {
	setwd(raw.data.dir);
	if(use.targets.file == TRUE){
		targets <- try(readTargets(targets.file), TRUE);
		if(!is.list(targets))
			stop("error: target file format is not correct!");
		slides <- targets$FileName;
		}
	
	##### assign spot weight #####
	if(identical(wt.fun, "spot")) RG <<- try(read.maimages(slides, source=MA.File.Format, wt.fun=wtarea(wt.para)), TRUE)
	else if(identical(wt.fun, "genepix")) RG <<- try(read.maimages(slides, source=MA.File.Format, wt.fun=wtflags(wt.para)), TRUE)
	else if(identical(wt.fun, "quantarray")) RG <<- try(read.maimages(slides, source=MA.File.Format, wt.fun=wtIgnore.Filter), TRUE)
	else if(identical(MA.File.Format, "user.defined")) RG <<- try(My.read.GPR.images(slides, source="user.defined"), TRUE)
	else RG <<- try(read.maimages(slides, source=MA.File.Format, wt.fun=NULL), TRUE);
	
	if(!is.list(RG)) stop("error: microarray intensity file name or format is not correct!");
	Num.of.Slides<<-length(slides);
	setwd(wd);
	}

#####################################################
############ read genome location file ##############
#####################################################

readGeneLoc <- function() {
	if(genome.sorting == TRUE){
		chr.position <<- try(read.delim(location.file, header=TRUE),TRUE);
		if(!is.list(chr.position)) stop("error: genome/chromosome location file format is not correct!");
		}
	}

#####################################################
################ experiment design  #################
#####################################################

makeDesign <- function() {
	if(simple.design == TRUE){
		designs <- array(c(design.ch1, design.ch2), dim=c(Num.of.Slides,2));
		if(is.list(contrast.design) && length(contrast.design) >= 1){
				designs <- mkDesign(designs, comp=contrast.design)
			design <<- designs$design
			contrast.matrix <<- designs$contrast

			design.num <<- length(contrast.design)
			contrast.name <<- mycontrastname(contra=contrast.design, format="list")
		}else{
			if(design.ref == TRUE)	ref.u <- "ref"
			else ref.u <- "ctrl"
			exp.design <<- mkDesign(targets=designs, ref=ref.u, comp=list(c("exp","ctrl")))
			design <<- exp.design$design
			contrast.matrix <<- exp.design$contrast
				design.num <<-	1
				contrast.name <<- c("exp-ctrl")
		}
	}else{
		design <<- read.delim(design.file);
		if(is.list(contrast.design) && length(contrast.design) >= 1) 
			contrast.design <<- as.list(c("", contrast.design))
		else	contrast.design <<- as.list(c("", colnames(design)))
			
		contrast.matrix <<- my.makeContrasts(contrast=contrast.design,levels=design);	
		design.num <<- length(contrast.design) - 1
		contrast.name <<- mycontrastname(contra=contrast.design, format="vector")
	}
	if((!is.vector(design) && is.array(design) && !is.list(design) && !is.matrix(design)) || 
		(!is.vector(contrast.matrix) && !is.array(contrast.matrix) && !is.list(contrast.matrix) && !is.matrix(contrast.matrix)))
		stop("error: experiment design is not correct!");
}

# prepare data: background substract and normalization
prepareData <- function() {

	#####################################################
	############# non-affymetrix array data  ###########
	#####################################################

	########### read gal file and array layout  #########
	
	if (gal.file.name != '')
		if(identical(gal, "GAL")) RG$genes <<- try(readGAL(gal.file.name), TRUE)
		else if(identical(gal, "udefined")) RG$genes <<- try(read.delim(gal.file.name, header=TRUE), TRUE);
	if (is.null(RG$genes)) stop("error: No gene list information found! <p>Perhaps you can offer a GAL file?")
	if(!is.list(RG$genes))	stop("error: gene list file format is not correct!");

	# for agilent data, we need modify the genes data:
	if (is.null(RG$genes$Block)) RG$genes$Block <<- 1
	if (is.null(RG$genes$Column)) RG$genes$Column <<- RG$genes$Col
	if (is.null(RG$genes$ID))
		if (!is.null(RG$genes$ProbeUID)) RG$genes$ID <<- RG$genes$ProbeUID
		else RG$genes$ID <<- seq(RG$genes$Row)
	if (is.null(RG$genes$Name))
		if (!is.null(RG$genes$GeneName)) RG$genes$Name <<- RG$genes$GeneName
		else if (!is.null(RG$genes$ProbeName)) RG$genes$Name <<- RG$genes$ProbeName
		else RG$genes$Name <<- as.character(RG$genes$ID)

	RG$printer <<- getLayout(RG$genes);
	Num.of.spots<-RG$printer$ngrid.r * RG$printer$ngrid.c * RG$printer$nspot.r * RG$printer$nspot.c;
	Num.of.spots.per.array<<- nrow(RG$R)/Num.of.Dup;
	
	################ assign spot types  #################
	
	if(spot.types == TRUE)	types <- try(readSpotTypes(spot.types.files), TRUE)
	else types <- data.frame("SpotType"="all spots", "ID"="*", "Name"="*", "Color"="black");
	if(!is.list(types))	stop("error: spot type file format is not correct!");
	RG$genes$Status <<- try(controlStatus(types, RG$genes), TRUE);
		
	############   background subtraction  ##############
	
	RG.before.bg.Sub <<- RG;
	RG <<- try(backgroundCorrect(RG, method=BG.Sub), TRUE);
	
	########## within-array normalization  ##############
		
	MA.before.WAN <<- try(normalizeWithinArrays(RG, RG$printer, method="none", bc.method="none"),TRUE);
	if(identical(WithinArray.Nor, "composite")){
		control.spots <- try(scan(controlspots), TRUE);
		if(!is.vector(control.spots)) stop("error: control spots file does not exist or file format is not correct!"); 
	}
	if(is.null(MA.before.WAN$weights)){
		weights <- MA.before.WAN$M
		f <- is.finite(weights)
		weights[f] <- 1
		MA.before.WAN$weights <- weights
	}

	MA <<- try(normalizeWithinArrays(MA.before.WAN, RG$printer, weights=MA.before.WAN$weights, method=WithinArray.Nor, controlspots=control.spots, bc.method="none"),TRUE);

	######### between-array normalization  ##############
	
	if(identical(BetweenArra.Nor, "vsn")){
		MA.before.BAN <<- MA.before.WAN;
		MA <<- try(normalizeBetweenArrays(RG.before.bg.Sub, method = BetweenArra.Nor),TRUE); }
	else if(identical(BetweenArra.Nor, "pca")){
		MA.before.BAN <<- MA.before.WAN;
		RG.pca.data <- cbind(RG$G, RG$R);
		RG.pca <- try(NormPCA(RG.pca.data, sig_level=pca.sig.level, ave_num=pca.ave.num));
		RG$G <<- RG.pca$ND[,c(1:Num.of.Slides)];
		RG$R <<- RG.pca$ND[,c((Num.of.Slides+1):(2*Num.of.Slides))];
		MA <<- MA.RG(RG); }
	else if(identical(BetweenArra.Nor, "none")){
		MA.before.BAN<<-MA;
		MA <<- MA
	}else{
		MA.before.BAN<<-MA;
		MA <<- try(normalizeBetweenArrays(MA, method = BetweenArra.Nor),TRUE); }
	if(!is.list(MA) && !is.array(MA)) stop("error: microarray data can not be normalized!");
	}

################################################################################
################################ Data Analysis   ###############################
################################################################################

#####################################################
############ export normalized data  ################
#####################################################

normalizeOnly <- function() {
	All.Results.vector<-c(RG$genes$Block[1:Num.of.spots.per.array], RG$genes$Row[1:Num.of.spots.per.array], RG$genes$Column[1:Num.of.spots.per.array], RG$genes$ID[1:Num.of.spots.per.array], RG$genes$Name[1:Num.of.spots.per.array]);
	All.Results.matrix<-matrix(All.Results.vector, ncol=5);
	col.names<-c();
	for( i in 1: length(slides)){
		if(Num.of.Dup > 1){
			arrayA<-array(c(MA$A[,i]), dim=c(Num.of.spots.per.array, Num.of.Dup));
			A<-apply(arrayA, 1, mean,na.rm=TRUE); 
			arrayM<-array(c(MA$M[,i]), dim=c(Num.of.spots.per.array, Num.of.Dup));
			M<-apply(arrayM, 1, mean,na.rm=TRUE); 
		}else{
			A <- MA$A[,i]
			M <- MA$M[,i]			
		}
		if(identical(design.ch1[i], "exp")){
				M <- (-M); 			}

		All.Results.matrix <- cbind(All.Results.matrix, M, A);
		col.names<-c(col.names, paste(slides[i],".M",sep=""), paste(slides[i],".A",sep=""));
		}
	colnames(All.Results.matrix)<-c("Block", "Row", "Column", "ID", "Name",col.names);
	write.table(All.Results.matrix, sep="\t", result.norm.data)
	}

#####################################################
############ statistical analysis  ##################
#####################################################

simpleDesign <- function() {
	fit <- try(lmFit(MA$M, design,ndups=Num.of.Dup, spacing=spacing, correlation=corr$consensus.correlation), TRUE);
	eb <- try(eBayes(fit), TRUE);
	if(!is.list(fit) || !is.list(eb)) stop("error: statistical analysis can not be executed!");
	if(design.ref == TRUE){
		M<-fit$coef[,2];		p<-eb$p.value[,2];
		t<-eb$t[,2];		B<-eb$lods[,2]; }
	else{
		M<-fit$coef;		p<-eb$p.value;
		t<-eb$t;		B<-eb$lods;	}
	A<-MA$A;
	A<-apply(A, 1, mean, na.rm=TRUE);
	if(Num.of.Dup > 1){
		arrayA<-array(A, dim=c(Num.of.spots.per.array, Num.of.Dup));
		A<-apply(arrayA, 1, mean, na.rm=TRUE); }
	my.fdr <- try(splosh(pvals=p), TRUE);
	if(!is.list(my.fdr)) stop("error: false dicovery rate calculation can not be executed!");
	cgh <- rep(NA, Num.of.spots.per.array);	

	#===========  Draw Chart for final Results  ============================
	if (plot.analyzed.results == TRUE) 
		try(plot.results(M=M,A=A,B=B,p=p,chart.name=paste(graphDir, Final.Chart.Title, '.png', sep='')), TRUE);
	
	#===========  Draw Chart for chromosome location  ======================
	if(genome.sorting == TRUE){
		M.chr <- M[chr.position[,1]];
		cgh.genes <- try(plot.chrom(ratio=M.chr, chr=chr.position[,2],main="",genome = genomes,
			genome.no=genome.no, a=span, c.thre=chr.thre.1, p.thre=chr.thre.2, 
			lim1 = chr.lim.1, lim2 = chr.lim.2,
			imagename=paste(graphDir, Location.Chart.Title, '.png', sep='')), TRUE);
		if(is.vector(cgh.genes)) cgh <- replace(cgh, chr.position[,1], cgh.genes);
		}
	#===========  write data  ==============================================	
	All.Results.vector<-c(RG$genes$Block[1:Num.of.spots.per.array], RG$genes$Row[1:Num.of.spots.per.array], RG$genes$Column[1:Num.of.spots.per.array], RG$genes$ID[1:Num.of.spots.per.array], RG$genes$Name[1:Num.of.spots.per.array], M, A, t, p, B, my.fdr$cfdr,my.fdr$fp,my.fdr$fn,cgh);
	All.Results.matrix <- try(matrix(All.Results.vector, ncol=14), TRUE);
	if(!is.matrix(All.Results.matrix))	stop("error: results file format is not correct!");
	colnames(All.Results.matrix)<-c("Block", "Row", "Column", "ID", "Name", "M", "A", "t", "P", "B", "fdr", "fp", "fn","CGH");
	if(identical(Output.Format, "SpotID"))	sort.index <- sort(c(1:Num.of.spots.per.array), index.return = TRUE)
	else if(identical(Output.Format, "sort.M"))	sort.index <- sort(M, index.return = TRUE)
	else if(identical(Output.Format, "sort.p")) sort.index <- sort(p, index.return = TRUE)
	else if(identical(Output.Format, "sort.B")) sort.index <- sort(B, index.return = TRUE)
	All.Results.matrix <- All.Results.matrix[sort.index$ix,];
	write.table(All.Results.matrix, sep = "\t", file=paste(result.analyzed.data, '.txt', sep=''));
	}

complexDesign <- function() {
	fit <- try(lmFit(MA$M, design,ndups=Num.of.Dup, spacing=spacing, correlation=corr$consensus.correlation), TRUE);
	fit2 <- try(contrasts.fit(fit, contrast.matrix), TRUE);
	fit2 <- try(eBayes(fit2), TRUE);

	if(!is.list(fit2)) stop("error: statistical analysis can not be executed!");
	if(Num.of.Dup == 1){		A<-MA$A;
	}else{	A <- array(c(MA$A), c(Num.of.spots.per.array, (length(MA$A[1,])*Num.of.Dup)))	}
	A<-apply(A, 1, mean, na.rm=TRUE);

	for ( i in 1: design.num){
		if(design.num == 1){
			M<-fit2$coef;			p<-fit2$p.value;
			t<-fit2$t;			B<-fit2$lods; }
		else {
			M<-fit2$coef[,i];			p<-fit2$p.value[,i];
			t<-fit2$t[,i];			B<-fit2$lods[,i]; }

		my.fdr <- try(splosh(pvals=p), TRUE);
		if(!is.list(my.fdr)) stop("error: false dicovery rate calculation can not be executed!");
		cgh <- rep(NA, Num.of.spots.per.array);
		
		#===========  Draw Chart for final Results  ====================
		if (plot.analyzed.results == TRUE) {
			chart.name<-paste(graphDir, Final.Chart.Title,mypaste(".comparison.",i, design.num, sep=''), '.', contrast.name[i], '.png',sep="");
			try(plot.results(M=M,A=A,B=B,p=p,chart.name=chart.name), TRUE);
			}
		
		#===========  Draw Chart for chromosome location  ==============
		if(genome.sorting == TRUE){
			M.chr <- M[chr.position[,1]];
			cgh.genes <- try(plot.chrom(ratio=M.chr, chr=chr.position[,2],genome = genomes,
				genome.no=genome.no, a=span, c.thre=chr.thre.1, p.thre=chr.thre.2, 
				lim1 = chr.lim.1, lim2 = chr.lim.2,plot.cgh=plot.cgh,
				imagename=paste(graphDir, Location.Chart.Title, mypaste('.', i, design.num, sep=''), '.', contrast.name[i], '.png', sep='')
				), TRUE);
			if(is.vector(cgh.genes)) cgh <- replace(cgh, chr.position[,1], cgh.genes);
			}
			#===========  write data  ======================================
		file.name<-paste(result.analyzed.data,mypaste(".comparison.",i, design.num, sep=''), '.', contrast.name[i], '.txt', sep="");
		All.Results.vector<-c(RG$genes$Block[1:Num.of.spots.per.array], RG$genes$Row[1:Num.of.spots.per.array], RG$genes$Column[1:Num.of.spots.per.array], RG$genes$ID[1:Num.of.spots.per.array], RG$genes$Name[1:Num.of.spots.per.array], M, A, t, p, B, my.fdr$cfdr,my.fdr$fp,my.fdr$fn, cgh);
		All.Results.matrix <- try(matrix(All.Results.vector, ncol=14), TRUE);
		if(!is.matrix(All.Results.matrix)) stop("error: results file format is not correct!");
		colnames(All.Results.matrix)<-c("Block", "Row", "Column", "ID", "Name", "M", "A", "t", "P", "B", "fdr", "fp", "fn", "CGH");
		if(identical(Output.Format, "SpotID"))	sort.index <- sort(c(1:Num.of.spots.per.array), index.return = TRUE)
		else if(identical(Output.Format, "sort.M")) sort.index <- sort(M, index.return = TRUE)
		else if(identical(Output.Format, "sort.p")) sort.index <- sort(p, index.return = TRUE)
		else if(identical(Output.Format, "sort.B")) sort.index <- sort(B, index.return = TRUE)
		All.Results.matrix <- All.Results.matrix[sort.index$ix,];
		write.table(All.Results.matrix, sep = "\t", file=file.name);		
		}
	}

analyze <- function() { 
	##### replicate subarray within array #####
	if(Num.of.Dup > 1){
		if(spacing>Num.of.spots.per.array || spacing<1 || (as.integer(spacing)-spacing)!=0){ stop("error: the number of spacing is not unaccurate!");	}
		corr <<- try(duplicateCorrelation(MA, design, spacing=spacing, ndups=Num.of.Dup), TRUE);
		#RG$genes<<-RG$genes[1:Num.of.spots.per.array,];
		if(!is.list(RG)) stop("error: within-array replicate correlation can not be calculated!");
		}
	complexDesign()
	}	
	

################################################################################
###########################                             ########################
###########################   Draw other charts         ########################
###########################                             ########################
################################################################################

#####################################################
############ chart 1: arrayimage plot ###############
#####################################################

drawArrayImage <- function() {
	for (i in 1: Num.of.Slides)	{
		plot.name <- paste(graphDir, mypaste(Array.Chart.Title, i, Num.of.Slides, sep=''), '.png', sep='')
		bitmap(plot.name, width=12,height=8)
		par(mfcol=c(2,2), bg="white");
		title.01<-paste("Chip", i, "/Ch1", sep="");
		try(imageplot(log(RG.before.bg.Sub$R[,i]+10,2), RG$printer, 
			low="white",high="red", main=title.01,cex.main=1,font.main=2,cex.lab=2, cex.axis=1), TRUE);
		title.02<-paste("Chip", i, "/Ch1 Background", sep="");
		try(imageplot(log(RG.before.bg.Sub$Rb[,i]+10,2), RG$printer, 
			low="white", high="red", main=title.02,cex.main=1,font.main=2,cex.lab=2, cex.axis=1), TRUE);
		title.03<-paste("Chip", i, "/Ch2", sep="");
		try(imageplot(log(RG.before.bg.Sub$G[,i]+10,2), RG$printer, 
			low="white", high="green", main=title.03,cex.main=1,font.main=2,cex.lab=2, cex.axis=1), TRUE);
		title.04<-paste("Chip", i, "/Ch2 Background", sep="");
		try(imageplot(log(RG.before.bg.Sub$Gb[,i]+10,2), RG$printer, 
			low="white", high="green", main=title.04,cex.main=1,font.main=2,cex.lab=2, cex.axis=1), TRUE);
		dev.off()
		}
	}

#####################################################
########### chart 2: density plot ###################
#####################################################

drawDensity <- function() {
	bitmap(paste(graphDir, Density.Chart.Title, sep=''), res=144, pointsize=10, width=6, height=2)
	par(mfcol=c(1,3), bg="white");
	try(plotDensities(MA.before.WAN), TRUE);
	title(sub="unnormalized intensity");
	try(plotDensities(MA.before.BAN), TRUE);
	title(sub="within-array normalized intensity");
	try(plotDensities(MA), TRUE);
	title(sub="between-array normalized intensity");	
	dev.off()
	}

#####################################################
############# chart 3: MA plot  #####################
#####################################################

drawMA <- function() {
	for( i in 1: Num.of.Slides) { #length(slides)){
		bitmap(paste(graphDir, mypaste(MA.Chart.Title,i, Num.of.Slides, sep=''),
			'.png', sep=''), width=12,height=6)
		par(mfcol=c(1,2), bg="white");
		try(plotMA(RG[,i], main="before withinarray normalization",status=RG$genes$Status, 
			cex.main=2, font.main=2, cex.lab=2, cex.axis=1, cex=0.1), TRUE);
		try(plotMA(MA.before.BAN[,i], main="after withinarray normalization",status=RG$genes$Status,
			cex.main=2, font.main=2, cex.lab=2, cex.axis=1, cex=0.1), TRUE);
		dev.off()
		}
	}

#####################################################
########## chart 4: printtiploess plot ##############
#####################################################

drawPrinttiploess <- function() {
	for (i in 1: Num.of.Slides){
		plot.name <- paste(graphDir, mypaste(Printtip.Chart.Title, i, 
			Num.of.Slides, sep=''), '.png', sep='')
		bitmap(plot.name, width=12,height=12)
		par(mfcol=c(2,2))

		title.01<-paste("Chip", i, ", before print-tip lowess normalization", sep="");
		try(plot.print.tip.lowess(RG.before.bg.Sub, RG$printer, pch=16, cex=0.1, 
			image=i, main=title.01, cex.main=1, cex.lab=1, cex.axis=1, font.main=2), TRUE);
		
		title.02<-paste("Chip", i, ", after print-tip lowess normalization", sep="");
		try(plot.print.tip.lowess(RG.before.bg.Sub, RG$printer, pch=16, cex=0.1, 
			image=i, norm="p", main=title.02, cex.main=1, cex.lab=1, cex.axis=1, font.main=2), TRUE);

		title.03<-paste("Chip", i, ", before print-tip lowess normalization", sep="");
		try(plot.scale.box(MA.before.WAN$M[,1], RG$printer, main=title.03,
			col=rainbow(RG$printer$ngrid.r*RG$printer$ngrid.c), cex.main=1, cex.lab=1, cex.axis=1, font.main=2), TRUE);
			abline(0,0)

		title.04<-paste("Chip", i, ", after print-tip lowess normalization", sep="");
		try(plot.scale.box(MA.before.BAN$M[,1], RG$printer, main=title.04, 
			col=rainbow(RG$printer$ngrid.r*RG$printer$ngrid.c), cex.main=1, cex.lab=1, cex.axis=1, font.main=2), TRUE);
			abline(0,0);
   
		dev.off() 
		}
	}

#####################################################
########### chart 5: Scale box plot #################
#####################################################

drawBox <- function() {
	bitmap(paste(graphDir, Scale.Chart.Title,sep=''), width=12, height=5)
	par(mfcol=c(1,2), bg="white")
	try(boxplot(MA.before.BAN$M~col(MA.before.BAN$M), names=slides,
			main="Before between-array normalization", cex.main=2,font.main=2, cex.axis=1), TRUE);
		abline(0,0)
	try(boxplot(MA$M~col(MA$M), cex.main=2,font.main=2, cex.axis=1,
		names=slides, main="After between-array normalization"), TRUE);
		abline(0,0)
	dev.off()
	}

drawCharts <- function() {
	############# save R image parameter ################
	op<-par(no.readonly=TRUE)
	if(plot.arrayimage == TRUE) drawArrayImage()
	if(plot.density == TRUE) drawDensity()
	if(plot.MA == TRUE) drawMA()
	if(plot.printtiploess == TRUE && WithinArray.Nor == "printtiploess") drawPrinttiploess()
	if(plot.box == TRUE) drawBox()
	##########################Recover R chart parameters###################
	par(op)
	}


SampleCodeMain <- function() {
	addNamePrefix()
	readIntensity()
	readGeneLoc()
	makeDesign()
	prepareData()
	setwd(result.data.dir)
	if(analysis == 'normalization'){
		normalizeOnly()
		setwd(wd) }
	else {
		analyze()
		drawCharts() }
	setwd(wd)
	}

SampleCodeMain()
##########################    Change Back File Path      ##############
setwd(wd);
rm(list=ls());
################################################################################
#############################    End of code    ################################
################################################################################
