try: 
	import paras
	if paras.USE_PSYCO:
		import psyco
		from psyco.classes import *
except: pass

import string, math, cmath, re, types, weakref, sys
from types import *

MAX_PRECISION = 2048
PRECISION = 128
VARIANCE = 1e-10 #used in numerical calculation
VARIANCE_GMPY = 1e-10 #used in numerical calculation
PI_800S = "3.1415926535897932384626433832795028841971693993751058209749445923078164062862089986280348253421170679821480865132823066470938446095505822317253594081284811174502841027019385211055596446229489549303819644288109756659334461284756482337867831652712019091456485669234603486104543266482133936072602491412737245870066063155881748815209209628292540917153643678925903600113305305488204665213841469519415116094330572703657595919530921861173819326117931051185480744623799627495673518857527248912279381830119491298336733624406566430860213949463952247371907021798609437027705392171762931767523846748184676694051320005681271452635608277857713427577896091736371787214684409012249534301465495853710507922796892589235420199561121290219608640344181598136297747713099605187072113499999983729780499510597317328160963185"
LOG2_2048S = "0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687542001481020570685733685520235758130557032670751635075961930727570828371435190307038623891673471123350115364497955239120475172681574932065155524734139525882950453007095326366642654104239157814952043740430385500801944170641671518644712839968171784546957026271631064546150257207402481637773389638550695260668341137273873722928956493547025762652098859693201965058554764703306793654432547632744951250406069438147104689946506220167720424524529612687946546193165174681392672504103802546259656869144192871608293803172714367783"
E_2048S = "2.71828182845904523536028747135266249775724709369995957496696762772407663035354759457138217852516642742746639193200305992181741359662904357290033429526059563073813232862794349076323382988075319525101901157383418793070215408914993488416750924476146066808226480016847741185374234544243710753907774499206955170276183860626133138458300075204493382656029760673711320070932870912744374704723069697720931014169283681902551510865746377211125238978442505695369677078544996996794686445490598793163688923009879312773617821542499922957635148220826989519366803318252886939849646510582093923982948879332036250944311730123819706841154"

special_char = ["\\", "{", "}", "[", "]", "(", ")", "<", ">", "\"", ",", ".", ":"]  # These characters are transformed to "\0x??_", "\" should locate first
obj_str = map(lambda k:hex(ord(k))+"_", special_char)  #obj_str = [str(hex(ord(k)))+"_" for k in special_char]
# \ is used to note special characters
# {}[]()<>" are used as boundary of dict, list, tuple, instance, string
# , :  are used in seq, dict
# . are used for relationship of belong, eg, AA.bb.1

basic_hide_src = re.compile(r"\\ | { | }", re.X)
basic_hide_obj = lambda a: "\\"+hex(ord(a.group()))+"_" # a is a re match object with a single char
basic_show_src= re.compile(r"\\0x5c_ | \\0x7b_ | \\0x7d_", re.X)
basic_show_obj = lambda a: chr(int(a.group()[1:-1], 16)) # a is a re match object, match "\0x??_"

seq_hide_src = tuple_hide_src = list_hide_src = re.compile(r"\\ | { | } | ,", re.X)
seq_hide_obj = tuple_hide_obj = list_hide_obj = basic_hide_obj
seq_show_src = tuple_show_src = list_show_src = re.compile(r"\\0x5c_ | \\0x7b_ | \\0x7d_ | \\0x2c_", re.X)
seq_show_obj = tuple_show_obj = list_show_obj = basic_show_obj 

dict_hide_src = re.compile(r"\\ | { | } | , | :", re.X)
dict_hide_obj = basic_hide_obj
dict_show_src= re.compile(r"\\0x5c_ | \\0x7b_ | \\0x7d_ | \\0x2c_ | \\0x3a_", re.X)
dict_show_obj = basic_show_obj

inst_src = ["\\", "{", "}", ".", ":", "->"]
inst_obj = ["\\0x5c_", "\\0x7b_", "\\0x7d_", "\\0x2e_", "\\0x3a_", "\\0x2d_\\0x3e_"]
if sys.version >= "2.2":
	inst_hide_dict = dict(zip(inst_src, inst_obj)) # {"\\":"\\0x5c_", "{":"\\0x7b_", "}":"\\0x7d_", ".":"\\0x2e_", ":":"\\0x3a_", "->":"\\0x2d_\\0x3e_"}
	inst_show_dict = dict(zip(inst_obj, inst_src)) #{"\\":"\\0x5c_", "{":"\\0x7b_", "}":"\\0x7d_", ".":"\\0x2e_", ":":"\\0x3a_", "->":"\\0x2d_\\0x3e_"}
else:
	inst_hide_dict = {}
	inst_show_dict = {}
	for i in range(len(inst_src)):
		inst_hide_dict[inst_src[i]] = inst_obj[i]
		inst_show_dict[inst_obj[i]] = inst_src[i]
		
inst_hide_src = re.compile(r"\\ | { | } | \. | : | ->", re.X)
inst_hide_obj = lambda a: inst_hide_dict[a.group()]
inst_show_src= re.compile(r"\\0x5c_	| \\0x7b_ | \\0x7d_ | \\0x2e_ | \\0x3a_ | \\0x2d_\\0x3e_", re.X)
inst_show_obj = lambda a: inst_show_dict[a.group()]

oct_str = re.compile(r"[\+ \-]?\s*0[0-7]+") 
dec_str = re.compile(r"[\+ \-]?\s*\d+") 
hex_str = re.compile(r"[\+ \-]?\s*0[xX][\dA-Fa-f]+") 
long_str = re.compile(r"[\+\-]?\s*\d+[Ll]") # decimal long
int_str = re.compile(r"[\+ \-]?\s*\d+  |  [\+ \-]?\s*0[xX][\dA-Fa-f]+  |  [\+\-]?\s*\d+[Ll]", re.X)
pure_float_str = re.compile(r"[\+\-]?\s*(\d+(\.\d*)?|\d*\.\d+)") # -23.4
float_str = re.compile(r"[\+\-]?\s*(\d+(\.\d*)?|\d*\.\d+)([Ee][\+\-]?\d+)?") # 34.44e+4
str_str = re.compile(r"\"\"\".*\"\"\"|'''.*'''|\".*\"|'.*'", re.S)
list_str = re.compile(r"\[.*\]", re.S) # very simple, not too reliable, so limited to my own data
tuple_str = re.compile(r"\(.*\)", re.S)
seq_str = re.compile(r"\[.*\]|\(.*\)", re.S)
dict_str = re.compile(r"\{.*\}", re.S)
instance_str = re.compile(r"<.*>", re.S)

def IsWhatStr(value, what): # Check if the str is a integer
	value  = value.strip()
	m = what.match(value)
	if m and m.group() == value: return 1
	return 0

class Nn: pass
import weakref
WeakrefType = type(weakref.ref(IsWhatStr))

def IsOctStr(value): # Check if the str is a integer
	return IsWhatStr(value, oct_str)
def IsDecStr(value): # Check if the str is a integer
	return IsWhatStr(value, dec_str)
def IsHexStr(value): # Check if the str is a integer
	return IsWhatStr(value, hex_str)
def IsLongStr(value): # Check if the str is a long integer
	return IsWhatStr(value, long_str)
def IsIntStr(value): # Check if the str is a integer
	return IsWhatStr(value, int_str)
def IsFloatStr(value): # Check if the str is a float
	return IsWhatStr(value, float_str)
def IsStrStr(value): # Check if the str is a string surrounded by ""
	return IsWhatStr(value, str_str)
def IsListStr(value): 
	return IsWhatStr(value, list_str)
def IsTupleStr(value): 
	return IsWhatStr(value, tuple_str)
def IsSeqStr(value):
	return IsWhatStr(value, seq_str)
def IsDictStr(value): 
	return IsWhatStr(value, dict_str)
def IsInstanceStr(value): 
	return IsWhatStr(value, instance_str)

def TransStr(value, src=seq_hide_src, obj=seq_hide_obj):
	# value should be a string, src is a re object, obj is a function
	if value and src and obj:
		return re.sub(src, obj, value) #src.sub(obj, value)
	return value

def GetSimpleValue(value, master=None, convert_me=0, convert_sub=1, convert_src=basic_show_src, convert_obj=basic_show_obj, **kw):
	if convert_me: value  = TransStr(value, src = convert_src, obj=convert_obj)
	if value == "None": return None, "none"
	elif IsLongStr(value):
		value = long(value), "long"
	elif IsIntStr(value):
		value = int(value, 0), "integer"
	elif IsFloatStr(value):
		value = float(value), "float"
	elif IsStrStr(value):
		value = value[1:-1]
		if convert_sub: value  = TransStr(value, src = convert_src, obj=convert_obj)
		value = value, "string"
	else: value = value, "string"
	return value
def GetStrValue(value, master=None, convert_me=0, convert_sub=1, convert_src=basic_show_src, convert_obj=basic_show_obj, **kw):
	if convert_me: value  = TransStr(value, src = convert_src, obj=convert_obj)
	if value == "None": return None
	elif IsStrStr(value):
		value = value[1:-1]
		if convert_sub: value  = TransStr(value, src = convert_src, obj=convert_obj)
	return value
def GetSeq(value, master=None, convert_me=0, convert_sub=1, convert_src=seq_show_src, convert_obj=seq_show_obj, **kw): 
	if convert_me: value = TransStr(value, src=covert_src, obj=convert_obj)
	first, last = value[0], value[-1]
	rtn  = string.split(value[1:-1], ",")
	for i in range(len(rtn)): 
		if convert_sub: rtn[i] = TransStr(rtn[i], src=convert_src, obj=convert_obj)
		rtn[i] = GetValue(rtn[i], master=master, convert_me=convert_me, convert_sub=convert_sub, convert_src=convert_src, convert_obj=convert_obj, **kw)[0]
	if first == "(" and last == ")": rtn = tuple(rtn)
	return rtn
def GetList(value, **kw): GetSeq(value, **kw)
def GetTuple(value, **kw):	GetSeq(value, **kw)
def GetDict(value, master=None, convert_me=0, convert_sub=1, convert_src=dict_show_src, convert_obj=dict_show_obj, **kw):	
	if convert_me: value = TransStr(value, src=covert_src, obj=convert_obj)
	if value[0] == "{" and value[-1] =="}":
		dict = {}
	else: return {}
	content = string.split(value[1:-1], ",")
	for item in content:
		if string.find(item, ":") < 0: continue
		k,v = string.split(item, ":")
		if convert_sub:
			k,v = TransStr(k, src=convert_src, obj=convert_obj), TransStr(v, src=convert_src, obj=convert_obj),
		k, kc = GetValue(k, master=master, convert_me=convert_me, convert_sub=convert_sub, convert_src=convert_src, convert_obj=convert_obj, **kw)
		v, vc = GetValue(v, master=master, convert_me=convert_me, convert_sub=convert_sub, convert_src=convert_src, convert_obj=convert_obj, **kw)
		if (kc == "instance" and not k) or (vc == "instance" and not v): 
			continue # skip any object that cannot be found for dict
		dict[k] = v
	return dict
def GetInstance(value, master=None, convert_me=0, convert_sub=1, convert_src=inst_show_src, convert_obj=inst_show_obj, changed_names=None, mixed=0, get_instance=1):	
	if convert_me: value = TransStr(value, src=convert_src, obj=convert_obj)
	if master is not None:
		d  = ParseInstName(value)
		index = d["index"]
		if convert_sub:
			dc, dn = TransStr(d["class"], src=convert_src, obj=convert_obj), TransStr(d["name"], src=convert_src, obj=convert_obj)
		else:
			dc, dn  = d["class"], d["name"]
		if not get_instance:
			return dn #d["name"]
		if dc == "Icon": 
			obj = GetInstance(dn, master=master, changed_names=changed_names, mixed=mixed) #master.GetInstance(dn, master.particle_type)
			if hasattr(obj, "icon_index") and obj.icon_index.has_key(index): 
				index = obj.icon_index[index]
			try:
				if index >= len(obj.icons): index = -1
				obj = obj.icons[index]
			except:
				print obj.name, index, obj.icons #, old_i
				raise
		elif dc == "weakref":
			obj, objc = GetValue(dn, master=master, changed_names=changed_names, mixed=mixed) #GetInstance(dn, master=master) #master.GetInstance(dn, dc)
			if obj and type(obj) is not StringType:
				obj = weakref.ref(obj)
			else: obj = weakref.ref(Nn())
		else: obj = master.GetInstance(dn, dc, changed_names=changed_names, mixed=mixed)
		return obj
	return value
class_name_index = re.compile(r"((?P<class>.+?):)?(?P<name>.+?)(\.(?P<index>\d+))?\s*\Z")
def ParseInstName(value, **kw): #, index = "auto"):
	value = value.strip()
	if value[0] == "<" and value[-1] == ">": value = value[1:-1]
	r = class_name_index
	m = r.match(value)
	d  = m.groupdict()
	if not d["class"] and d.get("class1", None): d["class"] = d["class1"]
	if not d["name"]: d["name"] = d.get("name1", None) or d.get("name2", None)
	dn = d["name"]
	if dn and len(dn)>1 and dn[0]=="\"" and dn[-1]=="\"": d["name"] = dn[1:-1]
	i = d.get("index", None) or d.get("index1", None)
	if i: d["index"] = int(i)
	d["single_name"] = string.split(d["name"], "->")[-1]
	return d
def GetIndex(value):
	value = value.strip()
	if value[0] == "<" and value[-1] == ">": value = value[1:-1]
	try:
		idx =int(icon.split(".")[-1]) #[:-1])
		return idx
	except: return None
def GetComplexValue(value, **kw): #used after GetSimpleValue
	tp = type(value) 
	if tp is not StringType: return value, tp
	if IsSeqStr(value):
		rtn = GetSeq(value, **kw), "sequence"
	elif IsDictStr(value):
		rtn = GetDict(value, **kw), "dictionary"
	elif IsInstanceStr(value,):
		rtn = GetInstance(value, **kw), "instance"
	else: rtn = value, "string"
	return rtn
def GetValue(value, **kw):
	tp = type(value) 
	if tp is not StringType: return value, tp
	value = value.strip()
	rtn = GetSimpleValue(value, **kw)
	if rtn[0] != value: pass #return rtn
	elif IsSeqStr(value):
		return GetSeq(value, **kw), "sequence"
	elif IsDictStr(value):
		return GetDict(value, **kw), "dictionary"
	elif IsInstanceStr(value,):
		return GetInstance(value, **kw), "instance"
	else: return value, "string"
	return rtn
simple_types = {IntType:"integer", LongType:"long", FloatType:"float"}
def RecurGet(value, master, changed_names=None, mixed=0):
	tp = type(value)
	if simple_types.has_key(tp):
		return value, simple_types[tp]
	elif tp is types.StringType:
		rtn = GetSimpleValue(value, master=master)
		if value != rtn[0]: return rtn
		rtn = GetComplexValue(value, master=master, changed_names=changed_names, mixed=mixed)
		if value == rtn[0]: return rtn
		elif rtn[0] is None and rtn[1] == "instance": return rtn
		return RecurGet(rtn[0], master=master, changed_names=changed_names, mixed=mixed)
	elif tp is types.ListType:
		for k in range(len(value)): value[k] = RecurGet(value[k], master, changed_names=changed_names, mixed=mixed)[0]
		return value, "list"
	elif tp is types.TupleType:
		a = []
		for k in range(len(value)): a.append(RecurGet(value[k]), master, changed_names=changed_names, mixed=mixed)[0]
		return tuple(a), "tuple"
	elif tp is types.DictType:
		for k, v in value.items():
			rk, rkt = RecurGet(k, master, changed_names=changed_names, mixed=mixed)
			if rk is None and rkt == "instance":
				del value[k]
				continue
			rv, rvt = RecurGet(v, master, changed_names=changed_names, mixed=mixed)
			if rv is None and rvt == "instance":
				del value[k]
				continue
			if rk != k:
				del value[k]
			value[rk] = rv
		return value, "dictionary"
	else: return value, tp
	
def NumStr(obj, master=None, convert_me=0, convert_sub=0, convert_src=None, convert_obj=None, **kw): #, use_default=1):  
	rtn = repr(obj) #str(obj)
	if convert_me or convert_sub:
		#if use_default: convert_src, convert_obj = seq_hide_src, seq_hide_obj
		rtn  = TransStr(rtn, src = convert_src, obj=convert_obj)
	return rtn
def StrStr(obj, master=None, convert_me=0, convert_sub=1, convert_src=basic_hide_src, convert_obj=basic_hide_obj, **kw): #, use_default=1): 
	if convert_sub:
		obj = TransStr(obj, src = convert_src, obj=convert_obj)
	rtn = "\""+obj+"\"" #usr ""
	if convert_me:
		#if use_default: convert_src, convert_obj = seq_hide_src, seq_hide_obj
		rtn  = TransStr(rtn, src = convert_src, obj=convert_obj)
	return rtn
def SeqStr(obj, first="[", last="]", master=None, convert_me=0, convert_sub=1,
		convert_src=seq_hide_src, convert_obj=seq_hide_obj, **kw): #, use_default=1):  #use []
	a = []
	for k in obj:
		#if obj_list is not None and type(k) is InstanceType and k not in obj_list: continue
		itm = GetStr(k, master=master, **kw) #, convert_me = convert_sub, convert_sub = convert_sub, convert_src=dict_hide_src, convert_obj=dict_hide_obj)
		if itm is None: continue
		if convert_sub: itm = TransStr(itm, src=convert_src, obj=convert_obj)
		a.append(itm)
	rtn = first+string.join(a, ", ")+last #", ".join(a) #or string.join(a, ", ")
	if convert_me: 
		#if use_default: convert_src, convert_obj = seq_hide_src, seq_hide_obj
		rtn  = TransStr(rtn, src = convert_src, obj=convert_obj)
	return rtn
ListStr = SeqStr  #use []
def TupleStr(obj, master=None, convert_me=0, convert_sub=1, convert_src=seq_hide_src, convert_obj=seq_hide_obj, **kw): #, use_default=1):  
	return SeqStr(obj, first="(", last=")", convert_me=convert_me, convert_sub=convert_sub, convert_src = convert_src, convert_obj=convert_obj, **kw)
def DictStr(obj, master=None, convert_me=0, convert_sub=1, convert_src=dict_hide_src, convert_obj=dict_hide_obj, **kw): #, use_default=1):  #use {}
	a = []
	all_obj = kw.get("all_obj", None)
	for k,v in obj.items():
		tv = type(v)
		if tv is WeakrefType and v() is None: continue
		key, value = GetStr(k, master=master, **kw), GetStr(v, master=master, **kw)
		if convert_sub: key, value = TransStr(key, src=convert_src, obj=convert_obj), TransStr(value, src=convert_src, obj=convert_obj)
		a.append( key + " : " + value)
	rtn = "{"+ ", ".join(a) + "}"
	if convert_me: 
		rtn  = TransStr(rtn, src = convert_src, obj=convert_obj)
	return rtn
def InstanceStr(obj, master=None, convert_me=0, convert_sub=1, convert_src=inst_hide_src, convert_obj=inst_hide_obj, save_icon_in_container=0, **kw): #, use_default=1):  #use <>
	if obj.__class__.__name__ == "Icon": #use <object_type:object_name.icon_number>
		class_name = obj.master.__class__.__name__
		if save_icon_in_container:
			obj_name = obj.master.name
			index = str(obj.master_icon.master.particle_icons.index(obj))
		else:
			obj_name = master.GetName(obj.master, **kw)
			index = str(obj.master.icons.index(obj))
		if convert_sub:
			class_name = TransStr(class_name, src = convert_src, obj=convert_obj)
			obj_name = TransStr(obj_name, src = convert_src, obj=convert_obj)
			index = TransStr(index, src = convert_src, obj=convert_obj)
		rtn = "<Icon:<"+class_name+":\""+obj_name+"\">."+index+">"
	else: #elif obj.__class__.name in self.object_type: #use <object_type:object_name>
		if convert_sub:
			class_name = TransStr(obj.__class__.__name__, src = convert_src, obj=convert_obj)
			obj_name = TransStr(master.GetName(obj, **kw), src = convert_src, obj=convert_obj)
		else:
			class_name = obj.__class__.__name__
			obj_name = master.GetName(obj, **kw)
		rtn = "<"+class_name+":\""+obj_name+"\">"
	if convert_me:
		rtn = TransStr(rtn, src = convert_src, obj=convert_obj)
	return rtn # something like <Icon:<Metabolite:"..->Com2->ATP">.4>
def WeakrefStr(obj, master=None, convert_me=0, convert_sub=1, convert_src=basic_hide_src, convert_obj=basic_hide_obj, **kw): #, use_default=1):  #use <>
	competitor = obj()
	if not competitor: 
		rtn = "None"
	else: 
		rtn = GetStr(competitor, master=master, **kw)
	if convert_sub:
		rtn = TransStr(rtn, src=convert_src, obj=convert_obj)
	rtn = "<weakref:"+rtn+">"
	if convert_me:
		rtn = TransStr(rtn, src=convert_src, obj=convert_obj)
	return rtn  # something like <weakref:<Icon:<Metabolite:"..->Com2->ATP">.4>>

str_func = {StringType:StrStr, ListType:ListStr, TupleType:TupleStr,
	DictType:DictStr, LongType:NumStr, IntType:NumStr, FloatType:NumStr,
	InstanceType:InstanceStr, NoneType:lambda *k, **kw:"None",WeakrefType:WeakrefStr}
def GetStr(obj, **kw): #master=None, convert_me=0, convert_sub=1, convert_src=seq_hide_src, convert_obj=seq_hide_obj, use_default=1): 
	return str_func[type(obj)](obj, **kw) #master=master, convert_me=convert_me, convert_sub=convert_sub, convert_src=convert_src, convert_obj=convert_obj, use_default=use_default)

def SortSeq(seq, index=0, direction="up"):
	# sort something like [(3, "aa", "ca"), (0, "asa", "dsa"), (23, "dslk", "pp"), ...]
	if not index: 
		seq = seq[:]
		seq.sort()
		if direction == "down": seq.reverse()
		return seq
	ln = range(len(seq))
	if ln:
		idx = [(seq[i][index], i) for i in ln]
		idx.sort()
		if direction == "down": idx.reverse()
		seq = [seq[idx[i][1]] for i in ln]
	return seq

def SeqPrint(data, file=sys.stdout, column=10, sep="\t"): 
	i = 0
	for k in data: 
		i = i+1
		file.write(k),
		if column and i==column:
			i=0
			file.write("\n")
		elif sep: file.write(sep),
	if i: file.write("\n")

def SeqFolder(data, column=10, sep=", "):
	tp = type(data)
	if tp is TupleType:
		first, last = "(", ")"
	elif tp is ListType:
		first, last = "[", "]"
	elif tp is DictType:
		first, last = "{", "}"
	else: return str(data)
	
	seq = [first]
	i = 0
	if tp in (TupleType, ListType):
		for k in data:
			i = i+1
			seq.append(SeqFolder(k))
			seq.append(sep)
			if column and  i==column:
				i = 0
				seq.append("\n")
		if data and (len(data)>1 or tp is ListType): del seq[i and -1 or -2] # delete the last ", "
	else: # must be a dict
		column = column/2
		for k, v in data.items():
			i = i+1
			seq.append(SeqFolder(k)+" : "+SeqFolder(v))
			seq.append(sep)
			if column and i==column:
				i = 0
				seq.append("\n")
		if data: del seq[i and -1 or -2]
	seq.append(last)
	return "".join(seq)
		
def ImportGmpy():
	try: a = gmpy.mpf(1.0)
	except:
		try:
			import gmpy
			globals()["gmpy"] = gmpy
		except: pass

def UpdatePrecision(n):
	global PI, LOG2, E, PRECISION
	PI      = gmpy.mpf(PI_800S)
	LOG2 = gmpy.mpf(LOG2_2048S)
	E = gmpy.mpf(E_2048S)
	PRECISION = n

try:
	import sys
	if not sys.modules.has_key("gmpy"):
		HAS_GMPY = 0
	else:
		HAS_GMPY = 1
	import gmpy
	if not HAS_GMPY: 
		prec = PRECISION
		gmpy.set_minprec(prec)
	else: prec = gmpy.mpf(1.0).getprec()
	UpdatePrecision(prec)
	HAS_GMPY = 1
except: 
	PI = math.pi
	LOG2 = math.log(2)
	E = math.e

def Ljust(strs, width=None):
	strs = strs.split("\n")
	if width is None: width = max([len(a) for a in strs])
	return "\n".join([a.ljust(width) for a in strs])

def Rjust(strs, width=None):
	strs = strs.split("\n")
	if width is None: width = max([len(a) for a in strs])
	return "\n".join([a.rjust(width) for a in strs])

def Center(strs, width=None):
	strs = strs.split("\n")
	if width is None: width = max([len(a) for a in strs])
	return "\n".join([a.center(width) for a in strs])
	
def Splite(func, squence, retvalues=None): #divide a sequence into several sequences
	retdict={}
	if retvalues==None:
		retvalues=[]
		add=1
	else:
		add=0
		for k in retvalues: retdict[k]=[]
	for k in squence:
		value=apply(func, (k,))
		if value not in retvalues:
			if add:
				retdict[value]=[]
				retvalues.append(value)
			else: continue
		retdict[value].append(k)
	if add: retvalues.sort()
	retlist=[]
	for k in retvalues:
		retlist.append(retdict[k])
	return tuple(retlist)

def GetBriefNumStr(c, dot = 4): # c is from str(), return a brief number string
	if type(c) is not types.StringType: c = repr(c)
	e = c.find("e")
	if e<0:
		e = len(c)
		t = ""
	else:
		t = c[e:]
	c = c[:e]
	d = c.find(".")
	if d<0:
		f = c
		b = ""
	else:
		b = c[d:e]
		f =c[0:d]
	if len(b)>5:
		b = float(b)
		if len(str(b))>6:
			b = ("%."+str(int(dot))+"f") % b
		else: b = str(b)
		b = b[1:]
	c = f+b+t
	return c

def GetKeywordStr(file): # can be used when read *.dms files
	keyword=""
	while 1:
		c=file.read(1)
		if not c: break
		if c not in "={}": 
			if not keyword and c in string.whitespace: continue
			if c == "\n": break
			keyword=keyword+c
		else: break
	return string.strip(keyword)

def GetValueStr(file): # can be used when read *.dms files
	# if a string contains "{" or "}", they may cause mistakes.
	brace_num=0
	value=""
	while 1:
		c=file.read(1)
		if not c: break # if a string contains "\0", it may cause mistake.
		if c=="{":
			if brace_num !=0: value=value+c
			brace_num=brace_num+1
		elif c=="}":
			brace_num=brace_num-1
			if brace_num==0: break
			else: value=value+c
		else: value=value+c
	return string.strip(value)

def GetName_old(s):  # Get name and coefficient
	"""Get a name string from s, then return it"""
	for i in range(len(s)):
		if s[i] in "+<=":
			i=i-1
			break
		elif s[i]=="-":
			for j in range(i+1, len(s)):
				if s[j]==">":
					meet_operator=1
					break
				elif s[j] not in string.whitespace:
					meet_operator=0
					break
			if meet_operator:
				i=i-1
				break
	i=i+1
	return s[0:i]

def GetName(s):  # Get name and coefficient , has bugs, no use now
	"""Get a name string from s, then return it
	It allow names like H+
	"""
	for i in range(len(s)):
		if s[i] in "<=":
			i=i-1
			break
		elif s[i]=="-":
			for j in range(i+1, len(s)):
				if s[j]==">":
					meet_operator=1
					break
				elif s[j] not in string.whitespace:
					meet_operator=0
					break
			if meet_operator:
				i=i-1
				break
		elif s[i]=="+":
			for j in range(i+1, len(s)):
				if s[j] in string.whitespace:
					continue
				elif s[j]=="+":
					i = j-1
					break
				else:
					break
			break
	i=i+1
	return s[0:i]

def GetOperator(s): #No use now
	"""Get + or -> or <-> (Note at present it can not deal with <-
                     return a list with the first is the operator may contain whitespace the the second without those
                    """
	current_ops=[]
	for i in range(len(s)):
		if s[i] not in string.whitespace:
			if s[i] in "+=":
				if string.strip(s[0:i+1])=="+" :
					return [s[0:i+1],"+"]
				elif string.strip(s[0:i+1])=="=" :
					return [s[0:i+1],"="]
				else:
					raise EquationError("Something unexpected before the '+' operator")
					return ["",""]
			elif s[i] in "<->":
				current_ops.append(s[i])
				if s[i]==">":
					if len(current_ops)==3 and current_ops[0]=="<" and current_ops[1]=="-":
						return [s[0:i+1],current_ops[0]+current_ops[1]+current_ops[2]]
					elif len(current_ops)==2 and current_ops[0]=="-":
						return [s[0:i+1],current_ops[0]+current_ops[1]]
					else:
						raise EquationError("Something wrong with the operator '->' or '<->'")
						return ["",""]
				if len(current_ops)>=3:
					raise EquationError("Something wrong with the operator '->' or '<->'")
					return ["",""]
	return [s[0:],""]

def CoeName_old(metabolite_name): # sperate the coefficient and the name
	coefficient=None
	if len(metabolite_name)>0:
		if metabolite_name[0] in "123456789":
			is_coefficient=1
			for j in range(len(metabolite_name)):
				if metabolite_name[j] not in string.digits:
					if metabolite_name[j] not in string.whitespace: #coefficient must followed by whitespace because of names like 3PG, 2PG
						is_coefficient=0
					else: # check if the first non-space is , or -
						for k in range(j, len(metabolite_name)):
							if metabolite_name[k] not in string.whitespace:
								if metabolite_name[k] in ",-'`":
									is_coefficient=0
								break
					break
			if is_coefficient:
				coefficient=int(metabolite_name[0:j])
				metabolite_name=string.strip(metabolite_name[j:])
	return coefficient, metabolite_name

def CoeName(metabolite_name): # sperate the coefficient and the name
	metabolite_name = string.strip(metabolite_name)
	coefficient=None
	mat  = re.compile(r"\d+\s+(?=[^,\-`\'])").match(metabolite_name)
	if mat:
		coefficient = int(mat.group())
		metabolite_name =  metabolite_name[mat.end():]
	return coefficient, metabolite_name

initial_name = {"Metabolite":"M", "Enzyme":"E", "Reaction":"R", "Channel":"C", "FreeChannel":"F", "Compartment":"Com", "Membrane":"Mem", "Subsystem":"S", "Simulation":"Project"}
def GetUniqueName(object_type, existed_names=[], src_name=None, sep=""):
	if src_name is None:
		src_name = initial_name[object_type]
	elif src_name not in existed_names: return src_name
	src_name=src_name.strip()
	tail = ""
	if not src_name: src_name = initial_name[object_type]
	else:
		while src_name[-1] == "+":
			src_name=src_name[:-1]
			tail = tail+"+"
	i=1
	myname=src_name+sep+str(i)+tail
	while myname in existed_names:
		i=i+1
		myname=src_name+sep+str(i)+tail
	return myname

def E_gmpy():
	e = gmpy.mpf(2.0)
	prod_i = i = gmpy.mpz(1)
	last_e = 0
	f1 = gmpy.mpf(1)
	while e != last_e:
		last_e = e
		i = i+1
		prod_i = prod_i*i
		e = e+ f1/prod_i
	return 	e
	
def Agm_gmpy(a, b):
	"""
	The arithmetic-geometric mean (often abbreviated AGM) M(a,b) of two numbers a and b.
	see http://mathworld.wolfram.com/Arithmetic-GeometricMean.html
	"""
	if a < b: a, b = gmpy.mpf(b), gmpy.mpf(a)
	else: a, b = gmpy.mpf(a), gmpy.mpf(b)
	f2 = gmpy.mpf(2.0)
	d, d_old = a-b, gmpy.mpf(0.0)
	ds  = [d_old, d]
	while d: 
		a, b = (a+b)/f2, gmpy.fsqrt(a*b)
		d = a - b
		if d == d_old or d in ds:
			break
		else: 
			d_old = d
			ds.append(d)
	return a

def Log_mpfr(a):
	q = MAX_PRECISION
	p = q+5
	m = q
	while m:
		p += 1
		m >>= 1
	if p%32:
		p += 32-(p%32)
	exp = "%5.3e" % a
	e = exp.find("e")
	exp = int(exp[e+1:])
	m = (p+1)/2 - exp +1

	s = a*gmpy.mpf(pow(2L, m)) #python2.1 need 2L because it cannot convert a very big integer to a long integer
	return PI/(2*Agm_gmpy(1.0, 4.0/s) )- m*LOG2

def Log_gmpy(x): # used by pow_gmpy
	if x <= 0:
		raise "log cannot deal with a number no more than zero"
	old_y = 0
	y = x = prod_x = (x - gmpy.mpf(1.0))/(x+1)
	x = x*x
	i = gmpy.mpz(1)
	while y != old_y: #abs(y - old_y)/abs(y) >1e-5:
		i = i+2
		old_y =  y
		prod_x = prod_x*x
		y = y+prod_x/i
	return 2*y

def Pow_gmpy(x, y=None):
	if x == 0: return 0
	elif y is not None:
		zy = gmpy.mpz(y)
		if zy == y: result = pow(gmpy.mpf(x), y)
		elif x < 0:
			raise "negative number cannot be raised to a fractional power"
		else:
			result = Pow_gmpy(y*Log_mpfr(x))
		return result
	return Epow_gmpy(x)

def Epow_gmpy(x, precision=None):
	# calculate e^x
	if precision is None: precision = PRECISION
	
	x = gmpy.mpf(x, precision)
	prod_x = x
	old_y = 1
	new_y = 1+x
	i = gmpy.mpz(1)
	prod_i = gmpy.mpz(1)
	while new_y != old_y:
		old_y = new_y
		i = i+1
		prod_i = prod_i*i
		prod_x = prod_x*x
		new_y = new_y + prod_x/prod_i
	else:
		if new_y < 0 and precision*2 <= MAX_PRECISION:  #new_y cannot be less than 0
			return Epow_gmpy(x, precision*2)
	return new_y

def Sin_gmpy(x): #used by Cpow_gmpy
	y = x = prod_x = gmpy.mpf(x)
	x_2 = x*x
	old_y = 0
	prod_i = i = gmpy.mpz(1)
	j = 0
	while y != old_y:
		old_y = y
		i = i+2
		j = (j+1)%2
		prod_i = prod_i * i * (i-1)
		prod_x = prod_x * x_2
		if j:
			y = y - prod_x/prod_i
		else:
			y = y + prod_x/prod_i
	return y

def Cos_gmpy(x): #used by Cpow_gmpy
	x = gmpy.mpf(x)
	x_2 = x*x
	y = prod_x = gmpy.mpf(1.0)
	old_y = 0
	i = gmpy.mpz(0)
	prod_i = gmpy.mpz(1)
	j = 0
	while y != old_y:
		old_y = y
		i = i+2
		j = (j+1)%2
		prod_i = prod_i * i * (i-1)
		prod_x = prod_x * x_2
		if j:
			y = y - prod_x/prod_i
		else:
			y = y + prod_x/prod_i
	return y

def Atan_gmpy(x): #used by Cpow_gmpy
	x = gmpy.mpf(x)
	if -1 < x <1:
		y = prod_x = x
		x_2 = x*x
		old_y = 0
		prod_i = i = gmpy.mpz(1)
		j = 0
		while y != old_y:
			old_y = y
			i = i+2
			j = (j+1)%2
			prod_i = prod_i * i * (i-1)
			prod_x = prod_x * x_2
			if j:
				y = y - prod_x/prod_i
			else:
				y = y + prod_x/prod_i
		return y
	elif x == 1:
		return PI/4
	elif x ==-1:
		return -PI/4
	elif x >= 1:
		y = PI/2-1/x
	else: y = -PI/2-1/x
	
	prod_x = x
	x_2 = x*x
	i = gmpy.mpz(1)
	old_y = y+1
	j =1
	while abs(y-old_y)>1e-9: #y != old_y:
		old_y = y
		i = i+2
		j = (j+1)%2
		prod_x = prod_x * x_2
		if j:
			y = y - 1/(prod_x*i)
		else:
			y = y + 1/(prod_x*i)
	return y

def Cpow(c, p): # a version of pow for plural number (pow(c, p))
	r=pow(math.sqrt(c.real*c.real+c.imag*c.imag), p)
	angler=math.atan(c.imag/c.real)*p
	if angler<0: angler=angler+math.pi
	return r*math.cos(angler)+r*math.sin(angler)*1j

def Cpow_gmpy(c, p): # a version of pow for plural number (pow(c, p))
	c = (gmpy.mpf(c[0]), gmpy.mpf(c[1]))
	r=Pow_gmpy(gmpy.fsqrt(c[0]*c[0]+c[1]*c[1]), p)
	angler=Atan_gmpy(c[1]/c[0])*p
	if angler<0: angler=angler+PI
	return r*Cos_gmpy(angler), r*Sin_gmpy(angler)

def Polynomial(f,x): #calculate the value of a polynomial  (f is the coefficient list) for a given x value
	n=len(f)
	value=f[0]
	if x:
		i=1
		for i in range(1,len(f)):
			value=value+f[i]*pow(x,i-1)
	return value

def Polynomial_gmpy(f,x): #calculate the value of a polynomial  (f is the coefficient list) for a given x value
	n=len(f)
	value=f[0]
	if x:
		i=1
		for i in range(1,len(f)):
			value=value+f[i]*Pow_gmpy(x,i-1)
	return value

def CubicEquation(f, direction = 1):  #Get the correct root of a Cubic equation, according to the reaction direction
	p=-f[2]*f[2]/3.0+f[1]
	q=2*f[2]*f[2]*f[2]/27.0-f[2]*f[1]/3.0+f[0]
	D=q*q/4+p*p*p/27
	if D>0: # one real root and two conjugate complex roots
		sqrt_d = math.sqrt(D)
		tmp=-q/2+sqrt_d
		if tmp>=0: delta_x=pow(tmp, 1.0/3)
		else: delta_x=-pow(-tmp, 1.0/3)

		tmp=-q/2-sqrt_d
		if tmp>=0: delta_x=delta_x+pow(tmp, 1.0/3)-f[2]/3.0
		else: delta_x=delta_x-pow(-tmp, 1.0/3)-f[2]/3.0
	elif D==0: # three real roots, of which two are equal;
		if q>=0: u=-pow(q/2, 1.0/3)
		else: u=pow(-q/2,1.0/3)
		x1=2*u- f[2]/3.0
		x2=-u-f[2]/3.0
		if direction>0:
			if x1*x2<0: delta_x=max(x1,x2)
			elif x1>=0 and x2>=0: delta_x=min(x1,x2)
			else: return 0
		else:
			if x1*x2<0: delta_x=min(x1,x2)
			elif x1<=0 and x2<=0: delta_x=max(x1,x2)
			else: return 0
	else: #D<0, three distinct real roots
		u1=Cpow(-q/2+cmath.sqrt(D), 1.0/3)
		a,b=u1.real,u1.imag
		x1=a+a-f[2]/3.0
		x2=-a+math.sqrt(3.0)*b-f[2]/3.0
		x3=-a-math.sqrt(3.0)*b-f[2]/3.0
		# sort to make x1<x2<x3
		if x2<x1:
			tmp=x2
			x2=x1
			x1=tmp
		if x3<x1:
			tmp=x3
			x3=x2
			x2=x1
			x1=tmp
		elif x3<x2:
			tmp=x3
			x3=x2
			x2=tmp
		if direction>0:
			if x1>0: delta_x=x1
			elif x2>0: delta_x=x2
			elif x3>0: delta_x=x3
			else: return 0
		else:
			if x3<0: delta_x=x3
			elif x2<0: delta_x=x2
			elif x1<0: delta_x=x1
			else: return 0

	if (direction>0 and delta_x>0) or (direction<0 and delta_x<0):
		return delta_x
	else: return 0
	
def CubicEquation_gmpy(f, direction = 1):  #Get the correct root of a Cubic equation, according to the reaction direction
	#This function has problem in solving some equation (e.g. cubic_problem.dms),
	# so DiMSim won't call it any more
	f0, f1, f2 = gmpy.mpf(f[0]), gmpy.mpf(f[1]), gmpy.mpf(f[2])
	g3 = gmpy.mpf(`3.0`)
	r3 = 1.0/g3
	p=-f2*f2/g3+f1
	q=2*f2*f2*f2/27.0-f2*f1/g3+f0
	D=q*q/4+p*p*p/27
	if D>0: # one real root and two conjugate complex roots
		sqrt_d = gmpy.fsqrt(D)
		tmp = -q/2+sqrt_d
		
		if tmp>=0: delta_x = Pow_gmpy(tmp, r3)
		else: delta_x = -Pow_gmpy(-tmp, r3)

		tmp = -q/2-sqrt_d
		if tmp>=0: delta_x = delta_x+Pow_gmpy(tmp, r3)-f2*r3
		else: delta_x = delta_x-Pow_gmpy(-tmp, r3)-f2*r3
	elif D==0: # three real roots, of which two are equal;
		if q>=0: u = -Pow_gmpy(q/2, r3)
		else: u = Pow_gmpy(-q/2, r3)
		x1=2*u- f[2]*r3
		x2=-u-f[2]*r3
		if direction>0:
			if x1*x2<0: delta_x=max(x1,x2)
			elif x1>=0 and x2>=0: delta_x=min(x1,x2)
			else: return 0
		else:
			if x1*x2<0: delta_x=min(x1,x2)
			elif x1<=0 and x2<=0: delta_x=max(x1,x2)
			else: return 0
	else: #D<0, three distinct real roots
		u1=Cpow_gmpy((-q/2, gmpy.fsqrt(-D)), r3)
		a, b = u1 #u1.real, u1.imag
		x1=a+a-f2*r3
		x2=-a+gmpy.fsqrt(g3)*b-f2*r3
		x3=-a-gmpy.fsqrt(g3)*b-f2*r3
		# sort to make x1<x2<x3
		if x2<x1:
			tmp=x2
			x2=x1
			x1=tmp
		if x3<x1:
			tmp=x3
			x3=x2
			x2=x1
			x1=tmp
		elif x3<x2:
			tmp=x3
			x3=x2
			x2=tmp
		if direction>0:
			if x1>0: delta_x=x1
			elif x2>0: delta_x=x2
			elif x3>0: delta_x=x3
			else: return 0
		else:
			if x3<0: delta_x=x3
			elif x2<0: delta_x=x2
			elif x1<0: delta_x=x1
			else: return 0
			
	if (direction>0 and delta_x>0) or (direction<0 and delta_x<0):
		return delta_x
	else: return 0

def PolyNR(f, start_x=0, variance =VARIANCE ): #Newton-Raphson(NR) method for polynomial  equation
	ff=[]
	for k in range(1, len(f)): ff.append(f[k]*k)
	new_x=start_x-Polynomial(f,start_x)/Polynomial(ff,start_x)
	new_y=Polynomial(f,new_x)
	while new_y>variance:
		new_x=new_x-new_y/Polynomial(ff, new_x)
		new_y=Polynomial(f, new_x)
	return new_x

def PolyNR_gmpy(f, start_x=0, variance =VARIANCE_GMPY ): #Newton-Raphson(NR) method for polynomial  equation
	ff=[]
	for k in range(1, len(f)): 
		f[k] = gmpy.mpf(f[k])
		ff.append(f[k]*k)
	new_x=start_x-Polynomial_gmpy(f,start_x)/Polynomial_gmpy(ff,start_x)
	new_y=Polynomial_gmpy(f,new_x)
	while new_y>variance:
		new_x=new_x-new_y/Polynomial_gmpy(ff, new_x)
		new_y=Polynomial_gmpy(f, new_x)
	return new_x

def PolyBiChop(f, x1, x2, y1=None, y2=None, variance = VARIANCE): #Binary chopping method for polynomial equation
	if y1==None: y1=Polynomial(f, x1)
	if not y1: return x1
	if y2==None: y2=Polynomial(f, x2)
	if not y2: return x2
	if y1*y2 > 0: return None # Binary chopping method is not suitable
	if y1>0: pos_x, neg_x = x1, x2
	else: pos_x, neg_x = x2, x1
	new_x=(pos_x+neg_x)/2.0
	new_y=Polynomial(f, new_x)
	while new_y > variance:
		if new_y>0: pos_x=new_x
		else: neg_x=new_x
		new_x=(pos_x+neg_x)/2.0
		new_y=Polynomial(f, new_x)
	return new_x

def PolyBiChop_gmpy(f, x1, x2, y1=None, y2=None, variance = VARIANCE_GMPY): #Binary chopping method for polynomial equation
	if y1 is None: y1=Polynomial_gmpy(f, x1)
	if not y1: return x1
	if y2 is None: y2=Polynomial_gmpy(f, x2)
	if not y2: return x2
	if y1*y2 > 0: return None # Binary chopping method is not suitable
	if y1>0: pos_x, neg_x = x1, x2
	else: pos_x, neg_x = x2, x1
	new_x=(pos_x+neg_x)/gmpy.mpf(2.0)
	new_y=Polynomial_gmpy(f, new_x)
	while new_y > variance:
		if new_y>0: pos_x=new_x
		else: neg_x=new_x
		new_x=(pos_x+neg_x)/2.0
		new_y=Polynomial_gmpy(f, new_x)
	return new_x

try:
	if paras.USE_PSYCO:
		for k in globals().values():
			if type(k) in [types.FunctionType, types.ClassType]:
				psyco.bind(k)
except: pass


import os
__override_functions__ = os.path.join(os.path.split(os.path.abspath(__file__))[0], 'toolfunc.hook')
if os.path.exists(__override_functions__): exec file(__override_functions__).read()
